<?php
session_start();

// Función para obtener el idioma actual
function getCurrentLanguage() {
    // Prioridad: parámetro GET > sesión > español por defecto
    if (isset($_GET['lang']) && in_array($_GET['lang'], ['es', 'en'])) {
        $_SESSION['language'] = $_GET['lang'];
        return $_GET['lang'];
    }
    
    if (isset($_SESSION['language']) && in_array($_SESSION['language'], ['es', 'en'])) {
        return $_SESSION['language'];
    }
    
    return 'es'; // idioma por defecto
}

// Función para obtener la URL con el parámetro de idioma
function getLanguageUrl($lang) {
    $currentUrl = $_SERVER['REQUEST_URI'];
    $parsedUrl = parse_url($currentUrl);
    
    // Si ya existe el parámetro lang, lo reemplazamos
    if (isset($parsedUrl['query'])) {
        parse_str($parsedUrl['query'], $params);
        $params['lang'] = $lang;
        $newQuery = http_build_query($params);
    } else {
        $newQuery = 'lang=' . $lang;
    }
    
    $baseUrl = $parsedUrl['path'];
    return $baseUrl . '?' . $newQuery;
}

// Cargar las traducciones
function loadTranslations($lang = 'es') {
    $translationsFile = dirname(__DIR__) . '/translations/' . $lang . '.php';
    if (file_exists($translationsFile)) {
        $translations = include $translationsFile;
        return is_array($translations) ? $translations : [];
    }
    
    // Si no existe el archivo, cargar español por defecto
    $defaultFile = dirname(__DIR__) . '/translations/es.php';
    if (file_exists($defaultFile) && $lang !== 'es') {
        $translations = include $defaultFile;
        return is_array($translations) ? $translations : [];
    }
    
    return [];
}

// Función principal para obtener texto traducido
function translate($key, $lang = null) {
    static $translations = null;
    static $currentLang = null;
    
    // Si no se especifica idioma, usar el idioma actual
    if ($lang === null) {
        $lang = getCurrentLanguage();
    }
    
    // Si es un idioma diferente o primera vez, cargar traducciones
    if ($translations === null || $currentLang !== $lang) {
        $translations = loadTranslations($lang);
        $currentLang = $lang;
    }
    
    // Si no hay traducciones cargadas, devolver fallback
    if (empty($translations)) {
        return getFallbackText($key);
    }
    
    // Buscar la clave en las traducciones
    $keys = explode('.', $key);
    $value = $translations;
    
    foreach ($keys as $k) {
        if (isset($value[$k])) {
            $value = $value[$k];
        } else {
            return getFallbackText($key); // Si no encuentra la clave, devolver fallback
        }
    }
    
    return is_string($value) ? $value : getFallbackText($key);
}

// Función de fallback para devolver texto por defecto si las traducciones fallan
function getFallbackText($key) {
    $fallbacks = [
        'nav.nosotros' => 'Nosotros',
        'nav.servicios' => 'Servicios', 
        'nav.proyectos' => 'Proyectos',
        'nav.simulador' => 'Simulador',
        'nav.reclutamiento' => 'Reclutamiento',
        'nav.contactanos' => 'Contáctanos',
        'common.back_to_top' => 'Ir al inicio',
        'common.inicio' => 'Inicio',
        'contact.title' => 'Contáctanos',
        'contact.quote' => '"El éxito en la construcción no solo depende de los planos, sino de la pasión y liderazgo del equipo."',
        'contact.email_us' => 'Escríbenos a',
        'contact.call_us' => 'o llama al:',
        'contact.form_title' => 'O déjanos tus datos y nosotros te contactamos',
        'contact.name' => 'Nombre',
        'contact.phone' => 'Número de teléfono',
        'contact.email' => 'Correo',
        'contact.send' => 'Enviar',
        'contact.social_networks' => 'Encuéntranos en redes',
    ];
    
    return isset($fallbacks[$key]) ? $fallbacks[$key] : $key;
}

// Función para obtener la ruta de imagen según el idioma
function getImagePath($imageName, $lang = null) {
    // Si no se especifica idioma, usar el idioma actual
    if ($lang === null) {
        $lang = getCurrentLanguage();
    }
    
    // Mapeo de imágenes específicas por idioma
    $imageMap = [
        'es' => [
            'ecosistema' => 'img/es/ecosistemaM2M.png',
            'sectores' => 'img/es/M2Msectores.png',
            'playbooks' => 'img/es/playbooks/'
        ],
        'en' => [
            'ecosistema' => 'img/en/ECOSISTEMA M2M [ENG].png',
            'sectores' => 'img/en/SECTORES M2M [ENG].png',
            'playbooks' => 'img/en/PLAYBOOKS INGLÉS/'
        ]
    ];
    
    // Si existe mapeo específico para esta imagen
    if (isset($imageMap[$lang][$imageName])) {
        return $imageMap[$lang][$imageName];
    }
    
    // Para playbooks individuales
    if (strpos($imageName, 'playbook') === 0) {
        $playbookNumber = str_replace('playbook', '', $imageName);
        if ($lang === 'en') {
            return 'img/en/PLAYBOOKS INGLÉS/PLAYBOOK ENG (' . $playbookNumber . ').png';
        } else {
            return 'img/es/playbooks/PLAYBOOK' . $playbookNumber . '.png';
        }
    }
    
    // Fallback: devolver ruta estándar
    return 'img/' . $imageName;
}

// Establecer el idioma actual
$currentLang = getCurrentLanguage();
?>